<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Reporte extends Model
{
    protected $fillable = [
        'codigo', 'tipo_id', 'lugar_id', 'area_id', 'turno_id',
        'descripcion', 'riesgo_id', 'consecuencia_id', 'es_anonimo',
        'reportado_por_user_id', 'reportado_por_texto',
        'firma_trabajador_path', 'accion_correctiva', 'supervisor_id',
        'firma_supervisor_path', 'plazo_horas_id', 'fecha_limite', 'estado_id',
    ];

    protected $casts = [
        'es_anonimo' => 'boolean',
        'fecha_limite' => 'datetime',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($reporte) {
            $yearMonth = now()->format('Ym');
            $count = self::whereYear('created_at', now()->year)
                         ->whereMonth('created_at', now()->month)
                         ->count() + 1;
            $reporte->codigo = 'RPT-' . $yearMonth . '-' . str_pad($count, 4, '0', STR_PAD_LEFT);
        });
    }

    public function tipo()
    {
        return $this->belongsTo(TipoReporte::class, 'tipo_id');
    }

    public function lugar()
    {
        return $this->belongsTo(Lugar::class, 'lugar_id');
    }

    public function area()
    {
        return $this->belongsTo(Area::class, 'area_id');
    }

    public function turno()
    {
        return $this->belongsTo(Turno::class, 'turno_id');
    }

    public function riesgo()
    {
        return $this->belongsTo(NivelRiesgo::class, 'riesgo_id');
    }

    public function consecuencia()
    {
        return $this->belongsTo(NivelConsecuencia::class, 'consecuencia_id');
    }

    public function reportadoPor()
    {
        return $this->belongsTo(User::class, 'reportado_por_user_id');
    }

    public function supervisor()
    {
        return $this->belongsTo(User::class, 'supervisor_id');
    }

    public function plazo()
    {
        return $this->belongsTo(PlazoHoras::class, 'plazo_horas_id');
    }

    public function estado()
    {
        return $this->belongsTo(EstadoReporte::class, 'estado_id');
    }

    public function fotos()
    {
        return $this->hasMany(ReporteFoto::class)->orderBy('orden');
    }

    public function getNombreReportanteAttribute(): string
    {
        if ($this->es_anonimo) {
            return 'Anónimo';
        }
        return $this->reportadoPor?->nombre_completo ?? $this->reportado_por_texto ?? 'Sin nombre';
    }

    public function scopeFilter($query, array $filters)
    {
        return $query
            ->when($filters['fecha_desde'] ?? null, fn($q, $v) => $q->whereDate('created_at', '>=', $v))
            ->when($filters['fecha_hasta'] ?? null, fn($q, $v) => $q->whereDate('created_at', '<=', $v))
            ->when($filters['tipo_id'] ?? null, fn($q, $v) => $q->where('tipo_id', $v))
            ->when($filters['lugar_id'] ?? null, fn($q, $v) => $q->where('lugar_id', $v))
            ->when($filters['area_id'] ?? null, fn($q, $v) => $q->where('area_id', $v))
            ->when($filters['turno_id'] ?? null, fn($q, $v) => $q->where('turno_id', $v))
            ->when($filters['riesgo_id'] ?? null, fn($q, $v) => $q->where('riesgo_id', $v))
            ->when($filters['estado_id'] ?? null, fn($q, $v) => $q->where('estado_id', $v))
            ->when($filters['supervisor_id'] ?? null, fn($q, $v) => $q->where('supervisor_id', $v))
            ->when($filters['buscar'] ?? null, fn($q, $v) => $q->where(function ($q) use ($v) {
                $q->where('codigo', 'like', "%{$v}%")
                  ->orWhere('descripcion', 'like', "%{$v}%")
                  ->orWhere('reportado_por_texto', 'like', "%{$v}%");
            }));
    }
}
