<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreReporteRequest;
use App\Models\Area;
use App\Models\EstadoReporte;
use App\Models\Lugar;
use App\Models\NivelConsecuencia;
use App\Models\NivelRiesgo;
use App\Models\Reporte;
use App\Models\TipoReporte;
use App\Models\Turno;
use App\Models\User;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class ReporteController extends Controller
{
    public function create()
    {
        return view('reportes.create', [
            'tipos' => TipoReporte::activos()->get(),
            'lugares' => Lugar::activos()->get(),
            'areas' => Area::activos()->get(),
            'turnos' => Turno::activos()->get(),
            'nivelesRiesgo' => NivelRiesgo::activos()->get(),
            'nivelesConsecuencia' => NivelConsecuencia::activos()->get(),
            'trabajadores' => User::activos()->orderBy('nombre_completo')->get(),
            'fechaActual' => now()->format('Y-m-d\TH:i'),
        ]);
    }

    public function store(StoreReporteRequest $request)
    {
        $estadoNuevo = EstadoReporte::where('nombre', 'Nuevo')->first();

        $reporte = Reporte::create([
            'tipo_id' => $request->tipo_id,
            'lugar_id' => $request->lugar_id,
            'area_id' => $request->area_id,
            'turno_id' => $request->turno_id,
            'descripcion' => $request->descripcion,
            'riesgo_id' => $request->riesgo_id,
            'consecuencia_id' => $request->consecuencia_id,
            'es_anonimo' => $request->boolean('es_anonimo'),
            'reportado_por_user_id' => $request->reportado_por_user_id,
            'reportado_por_texto' => $request->reportado_por_texto,
            'estado_id' => $estadoNuevo->id,
        ]);

        // Firma del trabajador
        if ($request->firma_trabajador) {
            $imageData = base64_decode(preg_replace('#^data:image/\w+;base64,#i', '', $request->firma_trabajador));
            $path = 'firmas/trabajador_' . $reporte->id . '.png';
            Storage::disk('public')->put($path, $imageData);
            $reporte->update(['firma_trabajador_path' => $path]);
        }

        // Fotos - guardar directamente sin comprimir (rápido y sin dependencia GD)
        if ($request->hasFile('fotos')) {
            foreach ($request->file('fotos') as $index => $foto) {
                try {
                    // Intentar comprimir con Intervention Image si GD está disponible
                    if (extension_loaded('gd')) {
                        $image = \Intervention\Image\Laravel\Facades\Image::read($foto);
                        $image->scaleDown(width: 1200);
                        $encoded = $image->toJpeg(quality: 70);
                        $path = 'reportes/' . $reporte->id . '/foto_' . ($index + 1) . '.jpg';
                        Storage::disk('public')->put($path, (string) $encoded);
                    } else {
                        // Sin GD: guardar directamente el archivo original
                        $extension = $foto->getClientOriginalExtension() ?: 'jpg';
                        $path = 'reportes/' . $reporte->id . '/foto_' . ($index + 1) . '.' . $extension;
                        $foto->storeAs('reportes/' . $reporte->id, 'foto_' . ($index + 1) . '.' . $extension, 'public');
                    }
                } catch (\Exception $e) {
                    // Fallback: guardar sin comprimir si hay cualquier error
                    Log::warning('Error al comprimir foto: ' . $e->getMessage());
                    $extension = $foto->getClientOriginalExtension() ?: 'jpg';
                    $path = 'reportes/' . $reporte->id . '/foto_' . ($index + 1) . '.' . $extension;
                    $foto->storeAs('reportes/' . $reporte->id, 'foto_' . ($index + 1) . '.' . $extension, 'public');
                }

                $reporte->fotos()->create([
                    'path' => $path,
                    'orden' => $index + 1,
                ]);
            }
        }

        return redirect()->route('reporte.confirmacion', $reporte);
    }

    public function confirmacion(Reporte $reporte)
    {
        return view('reportes.confirmacion', compact('reporte'));
    }
}
