<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class UsuarioController extends Controller
{
    public function index()
    {
        $usuarios = User::orderBy('nombre_completo')->paginate(20);
        return view('admin.usuarios.index', compact('usuarios'));
    }

    public function create()
    {
        return view('admin.usuarios.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nombre_completo' => 'required|string|max:255',
            'username' => 'required|string|max:100|unique:users|regex:/^[a-zA-Z0-9._-]+$/',
            'email' => 'nullable|email|unique:users',
            'password' => 'required|string|min:6|confirmed',
            'rol' => 'required|in:admin,supervisor,trabajador',
            'firma_png' => 'nullable|image|mimes:png|max:2048',
        ], [
            'username.regex' => 'El usuario solo puede contener letras, numeros, puntos, guiones y guiones bajos.',
            'username.unique' => 'Este nombre de usuario ya esta en uso.',
        ]);

        $user = User::create([
            'nombre_completo' => $validated['nombre_completo'],
            'username' => $validated['username'],
            'email' => $validated['email'] ?? null,
            'password' => Hash::make($validated['password']),
            'rol' => $validated['rol'],
        ]);

        if ($request->hasFile('firma_png')) {
            $path = $request->file('firma_png')->store('firmas/supervisores', 'public');
            $user->update(['firma_png_path' => $path]);
        }

        return redirect()->route('admin.usuarios.index')->with('success', 'Usuario creado.');
    }

    public function edit(User $usuario)
    {
        return view('admin.usuarios.edit', compact('usuario'));
    }

    public function update(Request $request, User $usuario)
    {
        $validated = $request->validate([
            'nombre_completo' => 'required|string|max:255',
            'username' => ['required', 'string', 'max:100', 'regex:/^[a-zA-Z0-9._-]+$/', Rule::unique('users')->ignore($usuario->id)],
            'email' => ['nullable', 'email', Rule::unique('users')->ignore($usuario->id)],
            'password' => 'nullable|string|min:6|confirmed',
            'rol' => 'required|in:admin,supervisor,trabajador',
            'activo' => 'boolean',
            'firma_png' => 'nullable|image|mimes:png|max:2048',
        ], [
            'username.regex' => 'El usuario solo puede contener letras, numeros, puntos, guiones y guiones bajos.',
            'username.unique' => 'Este nombre de usuario ya esta en uso.',
        ]);

        $usuario->update([
            'nombre_completo' => $validated['nombre_completo'],
            'username' => $validated['username'],
            'email' => $validated['email'] ?? null,
            'rol' => $validated['rol'],
            'activo' => $request->boolean('activo'),
        ]);

        if ($validated['password'] ?? null) {
            $usuario->update(['password' => Hash::make($validated['password'])]);
        }

        if ($request->hasFile('firma_png')) {
            if ($usuario->firma_png_path) {
                Storage::disk('public')->delete($usuario->firma_png_path);
            }
            $path = $request->file('firma_png')->store('firmas/supervisores', 'public');
            $usuario->update(['firma_png_path' => $path]);
        }

        return redirect()->route('admin.usuarios.index')->with('success', 'Usuario actualizado.');
    }

    public function destroy(User $usuario)
    {
        if ($usuario->id === auth()->id()) {
            return back()->with('error', 'No puedes eliminar tu propia cuenta.');
        }

        $usuario->update(['activo' => false]);
        return redirect()->route('admin.usuarios.index')->with('success', 'Usuario desactivado.');
    }
}
