<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Area;
use App\Models\EstadoReporte;
use App\Models\Lugar;
use App\Models\NivelConsecuencia;
use App\Models\NivelRiesgo;
use App\Models\PlazoHoras;
use App\Models\TipoReporte;
use App\Models\Turno;
use Illuminate\Http\Request;

class CatalogoController extends Controller
{
    private array $catalogos = [
        'tipos-reporte' => ['model' => TipoReporte::class, 'titulo' => 'Tipos de Reporte', 'extra_fields' => []],
        'lugares' => ['model' => Lugar::class, 'titulo' => 'Lugares', 'extra_fields' => []],
        'areas' => ['model' => Area::class, 'titulo' => 'Áreas', 'extra_fields' => []],
        'turnos' => ['model' => Turno::class, 'titulo' => 'Turnos', 'extra_fields' => []],
        'niveles-riesgo' => ['model' => NivelRiesgo::class, 'titulo' => 'Niveles de Riesgo', 'extra_fields' => []],
        'niveles-consecuencia' => ['model' => NivelConsecuencia::class, 'titulo' => 'Niveles de Consecuencia', 'extra_fields' => []],
        'plazos-horas' => ['model' => PlazoHoras::class, 'titulo' => 'Plazos (Horas)', 'extra_fields' => ['horas']],
        'estados-reporte' => ['model' => EstadoReporte::class, 'titulo' => 'Estados de Reporte', 'extra_fields' => []],
    ];

    public function index()
    {
        return view('admin.catalogos.index', ['catalogos' => $this->catalogos]);
    }

    public function manage(string $catalogo)
    {
        $config = $this->getCatalogoConfig($catalogo);
        $items = $config['model']::orderBy('orden')->get();

        return view('admin.catalogos.manage', [
            'items' => $items,
            'config' => $config,
            'catalogo' => $catalogo,
        ]);
    }

    public function store(Request $request, string $catalogo)
    {
        $config = $this->getCatalogoConfig($catalogo);

        $rules = ['nombre' => 'required|string|max:255', 'orden' => 'nullable|integer|min:0'];
        foreach ($config['extra_fields'] as $field) {
            $rules[$field] = 'required|integer|min:1';
        }

        $validated = $request->validate($rules);
        $validated['orden'] = $validated['orden'] ?? 0;

        $config['model']::create($validated);

        return redirect()->route('admin.catalogos.manage', $catalogo)->with('success', 'Registro creado.');
    }

    public function update(Request $request, string $catalogo, int $id)
    {
        $config = $this->getCatalogoConfig($catalogo);

        $rules = ['nombre' => 'required|string|max:255', 'orden' => 'nullable|integer|min:0'];
        foreach ($config['extra_fields'] as $field) {
            $rules[$field] = 'required|integer|min:1';
        }

        $validated = $request->validate($rules);
        $item = $config['model']::findOrFail($id);
        $item->update($validated);

        return redirect()->route('admin.catalogos.manage', $catalogo)->with('success', 'Registro actualizado.');
    }

    public function toggle(string $catalogo, int $id)
    {
        $config = $this->getCatalogoConfig($catalogo);
        $item = $config['model']::findOrFail($id);
        $item->update(['activo' => !$item->activo]);

        return redirect()->route('admin.catalogos.manage', $catalogo)->with('success', 'Estado actualizado.');
    }

    public function destroy(string $catalogo, int $id)
    {
        $config = $this->getCatalogoConfig($catalogo);
        $item = $config['model']::findOrFail($id);

        try {
            $item->delete();
            return redirect()->route('admin.catalogos.manage', $catalogo)->with('success', 'Registro eliminado.');
        } catch (\Exception $e) {
            return redirect()->route('admin.catalogos.manage', $catalogo)->with('error', 'No se puede eliminar: tiene registros asociados.');
        }
    }

    private function getCatalogoConfig(string $catalogo): array
    {
        if (!isset($this->catalogos[$catalogo])) {
            abort(404, 'Catálogo no encontrado.');
        }
        return $this->catalogos[$catalogo];
    }
}
